#!/bin/bash

LOG_FILE="/var/log/checker.log"
TTY="/dev/ttyS3"
TEE="tee --append $LOG_FILE $TTY"

function msg_checking() {
    PRINT_MSG=$1
    printf "connectivity to %-100s ......\n" "$PRINT_MSG" | $TEE
}

function msg_ok() {
    PRINT_MSG=$1
    printf "connectivity to %-100s ok      \n" "$PRINT_MSG" | $TEE
}

function msg_fail() {
    PRINT_MSG=$1
    printf "connectivity to %-100s failed  \n" "$PRINT_MSG" | $TEE
}

function ping_once() {
    PING_CHECK_HOST=$1
    ping -c 1 $PING_CHECK_HOST &> /dev/null
    if [ $? -eq 0 ]; then
        return 0
    else
        return 1
    fi
}

function nc_five_seconds() {
    NC_CHECK_PORT=$1
    NC_CHECK_HOST=$2
    NC_OUTPUT=$(nc -zv --wait 5 $NC_CHECK_HOST $NC_CHECK_PORT 2>&1)
    # When the port is blocked we get a timeout. in oher cases for most ports we don't have a listener
    # so we would also get and error status (Connection refused), this is why we check the string.
    if [[ $NC_OUTPUT == *"Ncat: TIMEOUT."* ]]; then
        return 1
    else
        return 0
    fi
}

function check() {
    COMAND=$1
    HOST=$2
    MSG=$3
    FULL_MSG="$MSG ($HOST)"
    msg_checking "$FULL_MSG"
    $COMAND "$HOST"
    if [ $? -eq 0 ]; then
        msg_ok "$FULL_MSG"
        return 0
    else
        msg_fail "$FULL_MSG"
        return 1
    fi
}

sudo yum install nc -y

GOOGLE_API_HOST="compute.googleapis.com"

echo "Waiting for up to 10 minutes for base connectivity before starting connectivity checks:" | $TEE
SECONDS=0
while [ $SECONDS -le 600 ]
do
  if ping_once "$GOOGLE_API_HOST"; then
    break
  fi
  sleep 10s
done

count=0

ZONE_URL="http://metadata.google.internal/computeMetadata/v1/instance/zone"
META_DATA_SERVICE_MSG="meta-data service ($ZONE_URL)"
msg_checking "$META_DATA_SERVICE_MSG"
ZONE_INFO=$(curl --silent $ZONE_URL --header "Metadata-Flavor: Google")
REGION_REGEX="^projects/[0-9]+/zones/(.*)-[a-z]+$"
if [[ "$ZONE_INFO" =~ $REGION_REGEX ]]; then
    REGION=$${BASH_REMATCH[1]}
    msg_ok "$META_DATA_SERVICE_MSG"
else
    msg_fail "$META_DATA_SERVICE_MSG"
    count=$((count + 1))
fi

check ping_once "$GOOGLE_API_HOST" "Compute Engine API"
count=$((count + $?))

TARGET_IP="${target_ip}"

check ping_once "$TARGET_IP" "internal cluster instance (ping)"
count=$((count + $?))

NC_PORTS=( 
    22     # ssh
    80     # http
    111    # rpcbind
    389    # ldap
    443    # https
    445    # smb
    636    # ldap ssl
    2049   # nfs
    3128   # callhome proxy
    3268   # ldap catalog
    3269   # ldap catalog ssl
    4000   # dnode internal
    4001   # dnode internal
    4100   # dnode internal
    4101   # dnode internal
    4200   # Cnode internal
    4201   # Cnode internal
    4420   # spdk target
    4520   # spdk target
    5000   # docker registry
    5200   # cnode internal dataenv
    5201   # cnode internal dataenv
    5551   # vms_monitor
    6000   # leader
    6001   # leader
    6126   # mlx sharpd
    7000   # dnode internal
    7001   # dnode internal
    7100   # dnode internal
    7101   # dnode internal
    8000   # mcvms
    9092   # kafka
    9090   # tabular
    20048  # mount
    20106  # nsm
    20107  # nlm
    20108  # nfs rquota
    49001  # replication
    49002  # replication
)
for NC_PORT in $${NC_PORTS[*]} 
  do
    check "nc_five_seconds $NC_PORT" "$TARGET_IP" "internal cluster instance (port $NC_PORT)"
    count=$((count + $?))
  done

ON_PREM_IP="${onprem_ip}"
if [[ -n "$ON_PREM_IP" ]]; then
    check ping_once "$ON_PREM_IP" "on-prem instance"
    count=$((count + $?))
fi

if [[ "$count" == "0" ]]; then
    echo "Connectivity check completed successfully" | $TEE
else
    echo "Connectivity check completed with errors" | $TEE
fi
