resource "google_compute_instance" "checker_target_instance" {
  name        = "${local.full_name}-target"
  description = "This instance is used as a target in internal connectivity checks"

  tags = concat(var.network_tags, local.voc_network_tags)

  machine_type = var.machine_type

  zone = var.zone

  scheduling {
    instance_termination_action = null
    min_node_cpus               = 0
    on_host_maintenance         = "MIGRATE"
    preemptible                 = false
    automatic_restart           = false
    provisioning_model          = "STANDARD"
  }

  boot_disk {
    initialize_params {
      image = var.image
    }
  }

  network_interface {
    subnetwork = var.subnetwork
  }

  metadata = {
    ssh-keys               = local.centos_public_key
    block-project-ssh-keys = true
    serial-port-enable     = true
  }
}

resource "google_compute_instance_template" "checker_instance_template" {
  name_prefix = "${local.full_name}-template"
  description = "This template is used to create a ${local.script_name} instance"

  tags = concat(var.network_tags, local.voc_network_tags)

  instance_description = "${local.script_name} instance"
  machine_type         = var.machine_type

  scheduling {
    instance_termination_action = null
    min_node_cpus               = 0
    on_host_maintenance         = "MIGRATE"
    preemptible                 = false
    automatic_restart           = false
    provisioning_model          = "STANDARD"
  }

  disk {
    source_image = var.image
    auto_delete  = true
    boot         = true
  }

  network_interface {
    subnetwork = var.subnetwork
  }

  metadata = {
    startup-script = templatefile("${path.module}/checker_script.sh", {
      onprem_ip = var.onprem_ip
      target_ip = google_compute_instance.checker_target_instance.network_interface[0].network_ip
    })

    ssh-keys               = local.centos_public_key
    block-project-ssh-keys = true
    serial-port-enable     = true
  }

  lifecycle {
    create_before_destroy = true
  }
}

resource "google_compute_health_check" "checker_instance_health_check" {
  name = "${local.full_name}-health-check"

  timeout_sec         = 10
  check_interval_sec  = 60
  healthy_threshold   = 2
  unhealthy_threshold = 2

  tcp_health_check {
    port = local.health_check_port
  }

  log_config {
    enable = false
  }
}

resource "google_compute_instance_group_manager" "checker_instance_group_manager" {
  name               = "${local.full_name}-instance-manager"
  base_instance_name = "${local.full_name}-instance"
  zone               = var.zone

  version {
    instance_template = google_compute_instance_template.checker_instance_template.self_link_unique
  }

  target_size = local.nodes_count

  auto_healing_policies {
    health_check      = google_compute_health_check.checker_instance_health_check.id
    initial_delay_sec = 600
  }
}

resource "time_sleep" "wait_for_health_check" {
  triggers = {
    template = google_compute_instance_template.checker_instance_template.creation_timestamp,
    group    = google_compute_instance_group_manager.checker_instance_group_manager.creation_timestamp,
  }

  depends_on = [
    google_compute_instance_group_manager.checker_instance_group_manager,
  ]

  create_duration = "5m"
}
